using System;
using System.Runtime.InteropServices;

namespace Sony
{
	namespace NP
	{
		public class Commerce
		{
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxCommerceIsBusy();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxCommerceCreateSession();

			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern ErrorCode PrxCommerceBrowseCategory(string categoryId);
			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern ErrorCode PrxCommerceBrowseProduct(string productId);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxCommerceDisplayDownloadList();
			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern ErrorCode PrxCommerceCheckout(string[] skuIDs, int count);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxCommerceVoucherInput();

			// Structure for returning category info.
			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct CommerceCategoryInfo
			{
				IntPtr _categoryId;					// The ID of the subcategory.
				IntPtr _categoryName;				// The name of the subcategory.
				IntPtr _categoryDescription;		// The detailed description of the subcategory.
				IntPtr _imageUrl;					// The image URL of the subcategory.
				public int countOfProducts;			// The number of products in the category, always 0 for sub-categories.
				public int countOfSubCategories;	// The number of subcategories, always 0 for sub-categories.
				public string categoryId { get { return Marshal.PtrToStringAnsi(_categoryId); } }
				public string categoryName { get { return Marshal.PtrToStringAnsi(_categoryName); } }
				public string categoryDescription { get { return Marshal.PtrToStringAnsi(_categoryDescription); } }
				public string imageUrl { get { return Marshal.PtrToStringAnsi(_imageUrl); } }
			};
			
			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern ErrorCode PrxCommerceRequestCategoryInfo(string categoryId);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxCommerceGetCategoryInfo(out CommerceCategoryInfo categoryInfo);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxCommerceGetSubCategoryInfo(int index, out CommerceCategoryInfo categoryInfo);

			// Structure for returning product info.
			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct CommerceProductInfo
			{
				public int purchasabilityFlag;		// A flag that indicates whether the product can be purchased (<c>SCE_NP_COMMERCE2_SKU_PURCHASABILITY_FLAG_XXX</c>).
				IntPtr _productId;				// The product ID.
				IntPtr _productName;			// The name of the product.
				IntPtr _shortDescription;		// A short description of the product.
				IntPtr _spName;					// The service provider name.
				IntPtr _imageUrl;				// The product image URL.
				IntPtr _price;					// The price of the product. This is formatted to include the currency code.	
				UInt64 releaseDate;				// The product release date.
				public string productId { get { return Marshal.PtrToStringAnsi(_productId); } }
				public string productName { get { return Marshal.PtrToStringAnsi(_productName); } }
				public string shortDescription { get { return Marshal.PtrToStringAnsi(_shortDescription); } }
				public string spName { get { return Marshal.PtrToStringAnsi(_spName); } }
				public string imageUrl { get { return Marshal.PtrToStringAnsi(_imageUrl); } }
				public string price { get { return Marshal.PtrToStringAnsi(_price); } }
			};

			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern ErrorCode PrxCommerceRequestProductList(string categoryId);
			[DllImport("UnityNpToolkit")]
			private static extern int PrxCommerceGetProductListInfoCount();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxCommerceGetProductListInfoItem(int index, out CommerceProductInfo productInfo);

			// Structure for returning detailed product info.
			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct CommerceProductInfoDetailed
			{
				public int purchasabilityFlag;		// A flag that indicates whether the product can be purchased (<c>SCE_NP_COMMERCE2_SKU_PURCHASABILITY_FLAG_XXX</c>).
				IntPtr _skuId;					// The SKU ID
				IntPtr _productId;				// The product ID.
				IntPtr _productName;			// The name of the product.
				IntPtr _shortDescription;		// A short description of the product.
				IntPtr _longDescription;		// A long description of the product.
				IntPtr _legalDescription;		// The legal description for the product.	
				IntPtr _spName;					// The service provider name.
				IntPtr _imageUrl;				// The product image URL.
				IntPtr _price;					// The price of the product. This is formatted to include the currency code.
				IntPtr _ratingSystemId;			// The ID of the rating system (for example: PEGI, ESRB).
				IntPtr _ratingImageUrl;			// The URL of the rating icon.
				UInt64	releaseDate;			// The product release date.
				public int numRatingDescriptors;		// The number of rating descriptors stored for this product.
				public string skuId { get { return Marshal.PtrToStringAnsi(_skuId); } }
				public string productId { get { return Marshal.PtrToStringAnsi(_productId); } }
				public string productName { get { return Marshal.PtrToStringAnsi(_productName); } }
				public string shortDescription { get { return Marshal.PtrToStringAnsi(_shortDescription); } }
				public string longDescription { get { return Marshal.PtrToStringAnsi(_longDescription); } }
				public string legalDescription { get { return Marshal.PtrToStringAnsi(_legalDescription); } }
				public string spName { get { return Marshal.PtrToStringAnsi(_spName); } }
				public string imageUrl { get { return Marshal.PtrToStringAnsi(_imageUrl); } }
				public string price { get { return Marshal.PtrToStringAnsi(_price); } }
				public string ratingSystemId { get { return Marshal.PtrToStringAnsi(_ratingSystemId); } }
				public string ratingImageUrl { get { return Marshal.PtrToStringAnsi(_ratingImageUrl); } }
			};

			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern ErrorCode PrxCommerceRequestDetailedProductInfo(string productId);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxCommerceGetProductInfoDetailed(out CommerceProductInfoDetailed productInfo);

			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct CommerceEntitlement
			{
				IntPtr _id;
				public int type;
				public int remainingCount;  // may be negative
				public int consumedCount;
				public UInt64 createdDate;
				public UInt64 expireDate;
				public string id { get { return Marshal.PtrToStringAnsi(_id); } }
			};

			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxCommerceRequestEntitlementList();
			[DllImport("UnityNpToolkit")]
			private static extern int PrxCommerceGetEntitlementListCount();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxCommerceGetEntitlementListItem(int index, out CommerceEntitlement entitlement);
			
			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern ErrorCode PrxCommerceConsumeEntitlement(string entitlementId, int consumeCount);


			public enum StoreIconPosition
			{
				Left = 0,
				Center = 1,
				Right = 2,
			};

			[DllImport("UnityNpToolkit")]
			private static extern bool PrxCommerceShowStoreIcon(bool show, StoreIconPosition position);

			[DllImport("UnityNpToolkit")]
			private static extern bool PrxCommerceGetLastError(out ResultCode result);
			
			public static bool GetLastError(out ResultCode result)
			{
				PrxCommerceGetLastError(out result);
				return result.lastError == ErrorCode.NP_OK;
			}

			// Is the word filter busy with a request.
			public static bool IsBusy()
			{
				return PrxCommerceIsBusy();
			}
			
			public static ErrorCode CreateSession()
			{
				return PrxCommerceCreateSession();
			}

			public static ErrorCode BrowseCategory(string categoryId)
			{
				return PrxCommerceBrowseCategory(categoryId);
			}
			
			public static ErrorCode BrowseProduct(string productId)
			{
				return PrxCommerceBrowseProduct(productId);
			}

			public static ErrorCode DisplayDownloadList()
			{
				return PrxCommerceDisplayDownloadList();
			}

			public static ErrorCode Checkout(string[] skuIDs)
			{
				return PrxCommerceCheckout(skuIDs, skuIDs.Length);
			}

			public static ErrorCode VoucherInput()
			{
				return PrxCommerceVoucherInput();
			}
			
			public static ErrorCode RequestCategoryInfo(string categoryId)
			{
				return PrxCommerceRequestCategoryInfo(categoryId);
			}
			
			public static CommerceCategoryInfo GetCategoryInfo()
			{
				CommerceCategoryInfo info = new CommerceCategoryInfo();
				PrxCommerceGetCategoryInfo(out info);
				return info;
			}
			
			public static CommerceCategoryInfo GetSubCategoryInfo(int index)
			{
				CommerceCategoryInfo info = new CommerceCategoryInfo();
				PrxCommerceGetSubCategoryInfo(index, out info);
				return info;
			}

			public static ErrorCode RequestProductList(string categoryId)
			{
				return PrxCommerceRequestProductList(categoryId);
			}

			public static CommerceProductInfo[] GetProductList()
			{
				int count = PrxCommerceGetProductListInfoCount();
				CommerceProductInfo[] products = new CommerceProductInfo[count];
				for(int i=0; i<count; i++)
				{
					PrxCommerceGetProductListInfoItem(i, out products[i]);
				}
				return products;
			}

			public static ErrorCode RequestDetailedProductInfo(string productId)
			{
				return PrxCommerceRequestDetailedProductInfo(productId);
			}

			public static CommerceProductInfoDetailed GetDetailedProductInfo()
			{
				CommerceProductInfoDetailed info = new CommerceProductInfoDetailed();
				PrxCommerceGetProductInfoDetailed(out info);
				return info;
			}

			public static ErrorCode RequestEntitlementList()
			{
				return PrxCommerceRequestEntitlementList();
			}

			public static CommerceEntitlement[] GetEntitlementList()
			{
				int count = PrxCommerceGetEntitlementListCount();
				CommerceEntitlement[] ents = new CommerceEntitlement[count];
				for(int i=0; i<count; i++)
				{
					PrxCommerceGetEntitlementListItem(i, out ents[i]);
				}
				return ents;
			}

			public static ErrorCode ConsumeEntitlement(string entitlementId, int consumeCount)
			{
				return PrxCommerceConsumeEntitlement(entitlementId, consumeCount);
			}

			public static bool ShowStoreIcon(StoreIconPosition position)
			{
				return PrxCommerceShowStoreIcon(true, position);
			}
			
			public static bool HideStoreIcon()
			{
				return PrxCommerceShowStoreIcon(false, StoreIconPosition.Center);
			}

			// Event delegates.
			public static event Messages.EventHandler OnSessionCreated;
			public static event Messages.EventHandler OnSessionAborted;
			public static event Messages.EventHandler OnGotCategoryInfo;
			public static event Messages.EventHandler OnGotProductList;
			public static event Messages.EventHandler OnGotProductInfo;
			public static event Messages.EventHandler OnGotEntitlementList;
			public static event Messages.EventHandler OnConsumedEntitlement;
			public static event Messages.EventHandler OnError;
			public static event Messages.EventHandler OnCheckoutStarted;
			public static event Messages.EventHandler OnCheckoutFinished;

			// Message processing.
			public static bool ProcessMessage(Messages.PluginMessage msg)
			{
				switch (msg.type)
				{
					case Messages.MessageType.kNPToolKit_CommerceSessionCreated:
						if (OnSessionCreated != null) OnSessionCreated(msg);
						return true;

					case Messages.MessageType.kNPToolKit_CommerceSessionAborted:
						if (OnSessionAborted != null) OnSessionAborted(msg);
						return true;

					case Messages.MessageType.kNPToolKit_CommerceGotCategoryInfo:
                        if (OnGotCategoryInfo != null) OnGotCategoryInfo(msg);
						return true;

					case Messages.MessageType.kNPToolKit_CommerceGotProductList:
                        if (OnGotProductList != null) OnGotProductList(msg);
						return true;

					case Messages.MessageType.kNPToolKit_CommerceGotProductInfo:
                        if (OnGotProductInfo != null) OnGotProductInfo(msg);
						return true;

					case Messages.MessageType.kNPToolKit_CommerceGotEntitlementList:
						if (OnGotEntitlementList != null) OnGotEntitlementList(msg);
						return true;

					case Messages.MessageType.kNPToolKit_CommerceConsumedEntitlement:
						if (OnConsumedEntitlement != null) OnConsumedEntitlement(msg);
						return true;

					case Messages.MessageType.kNPToolKit_CommerceError:
						if (OnError != null) OnError(msg);
						return true;

					case Messages.MessageType.kNPToolKit_CommerceCheckoutStarted:
						if (OnError != null) OnCheckoutStarted(msg);
						return true;

					case Messages.MessageType.kNPToolKit_CommerceCheckoutFinished:
						if (OnError != null) OnCheckoutFinished(msg);
						return true;
				}
				
				return false;
			}
			
		}
	}
}
